-- WinchHookPathFix.lua (FS25)
-- Purpose: allow custom (moddir/relative) treeHook filenames for Winch ropes.
-- Strategy: inject a baseDirectory when GIANTS calls ForestryHook:loadFromXML without one,
-- and (as a fallback) re-load the hook from Winch with self.baseDirectory after rope load.

local function log(fmt, ...)
    local msg = string.format("[WinchHookPathFix] " .. fmt, ...)
    if Logging and Logging.info then Logging.info("%s", msg) else print(msg) end
end

----------------------------------------------------------------
-- 1) Patch ForestryHook:loadFromXML to inject baseDirectory
----------------------------------------------------------------
if ForestryHook ~= nil and ForestryHook.loadFromXML ~= nil and not ForestryHook.__PathFixApplied then
    local _FH_loadFromXML = ForestryHook.loadFromXML

    function ForestryHook:loadFromXML(xmlFile, key, baseDirectory)
        if baseDirectory == nil then
            -- Try to resolve a sensible base directory from the owning vehicle
            if self.vehicle ~= nil and self.vehicle.baseDirectory ~= nil then
                baseDirectory = self.vehicle.baseDirectory
            elseif self.baseDirectory ~= nil then
                baseDirectory = self.baseDirectory
            end
        end
        return _FH_loadFromXML(self, xmlFile, key, baseDirectory)
    end

    ForestryHook.__PathFixApplied = true
    log("Patched ForestryHook:loadFromXML to inject baseDirectory when missing")
end

----------------------------------------------------------------
-- 2) Patch Winch:loadWinchRopeFromXML to pass baseDirectory to hook loader (fallback)
----------------------------------------------------------------
if Winch ~= nil and Winch.loadWinchRopeFromXML ~= nil and not Winch.__HookReloadPatched then
    local _W_loadRope = Winch.loadWinchRopeFromXML

    function Winch:loadWinchRopeFromXML(xmlFile, key, rope)
        local ok = _W_loadRope(self, xmlFile, key, rope)

        -- If the hook failed to resolve (common when only $data works), try again with baseDirectory
        if rope ~= nil and rope.hookData ~= nil then
            local hd = rope.hookData
            local hasXml = (hd.xmlFilename ~= nil and hd.xmlFilename ~= "") or (hd.filename ~= nil and hd.filename ~= "")
            -- If no xml path was parsed at all, (re)load with a baseDirectory so $moddir$/relative paths resolve
            if not hasXml then
                if self.baseDirectory ~= nil then
                    log("Hook reload with baseDirectory: %s", tostring(self.baseDirectory))
                else
                    log("Hook reload: baseDirectory unknown (will still try)")
                end
                -- GIANTS loader expects key .. ".treeHook"
                hd:loadFromXML(xmlFile, key .. ".treeHook", self.baseDirectory)
            end
        end

        return ok
    end

    Winch.__HookReloadPatched = true
    log("Patched Winch:loadWinchRopeFromXML to ensure hook reload with baseDirectory when needed")
end

log("Activated - custom $moddir$/relative treeHook filenames should now resolve.")


----------------------------------------------------------------
-- V2b: Ensure ALL rope sub-elements resolve paths with a robust baseDirectory
-- Covers: <mainRope>, <setupRope>, <marker>, and <treeHook>
----------------------------------------------------------------
local function _whpf_log(fmt, ...) 
    local msg = string.format("[WinchHookPathFix] " .. fmt, ...)
    if Logging and Logging.info then Logging.info("%s", msg) else print(msg) end
end

-- 1) Force a non-nil, *robust* baseDirectory into Winch:loadWinchRopeFromXML
if Winch ~= nil and Winch.loadWinchRopeFromXML ~= nil and not Winch.__RopeBaseDirPatched then
    local _W_loadRope = Winch.loadWinchRopeFromXML
    Winch.loadWinchRopeFromXML = function(self, xmlFile, key, baseDirectory, ...)
        -- Build a resilient base dir:
        local bd = baseDirectory
        if bd == nil or bd == "" then bd = self and self.baseDirectory or nil end
        if (bd == nil or bd == "") and xmlFile ~= nil and xmlFile.getDirectory ~= nil then
            bd = xmlFile:getDirectory()
        end
        if (bd == nil or bd == "") and self ~= nil and self.configFileName ~= nil then
            -- strip filename to directory
            bd = self.configFileName:gsub("[^/\\]+$", "")
        end
        if bd == nil then bd = "" end
        return _W_loadRope(self, xmlFile, key, bd, ...)
    end
    Winch.__RopeBaseDirPatched = true
    _whpf_log("Patched Winch:loadWinchRopeFromXML to pass a robust baseDirectory for all rope sub-elements")
end

-- 2) Fallback post-load normalizer (in case some fields were read manually without Utils.getFilename)
--    This pass gently re-roots any non-$ paths on known rope fields.
local function resolvePath(path, baseDir)
    if path == nil or path == "" then return path end
    if path:sub(1,1) == "$" then return path end -- $data or $moddir$ honored
    if Utils and Utils.getFilename then
        return Utils.getFilename(path, baseDir or "")
    end
    return path
end

local function normalizeRopeObjectPaths(self, rope, xmlFile)
    -- compute a solid baseDir once
    local baseDir = (self and self.baseDirectory) or ((xmlFile and xmlFile.getDirectory and xmlFile:getDirectory()) or (self and self.configFileName and self.configFileName:gsub("[^/\\]+$", "")) or "")
    -- Common fields we might see on rope / parts. Add more if your rope exposes them.
    local ropeFields = { "filename", "xmlFilename", "i3dFilename", "segmentFilename", "materialFilename", "textureFilename" }
    local function fixTable(t, bd)
        if type(t) ~= "table" then return end
        for _, k in ipairs(ropeFields) do
            if t[k] ~= nil and t[k] ~= "" then
                local fixed = resolvePath(t[k], bd)
                if fixed ~= t[k] then
                    _whpf_log("Re-rooted %s: %s -> %s", k, tostring(t[k]), tostring(fixed))
                    t[k] = fixed
                end
            end
        end
    end

    if type(rope) ~= "table" then return end
    -- Top-level rope fields
    fixTable(rope, baseDir)
    -- Known sub-objects used by various winch implementations
    fixTable(rope.mainRope, baseDir)
    fixTable(rope.setupRope, baseDir)
    fixTable(rope.treeHook, baseDir)
    fixTable(rope.hook, baseDir)          -- some mods use 'hook' instead of 'treeHook'
    -- Markers can be a table or an array of tables
    if rope.marker then
        if #rope.marker > 0 then
            for _, m in ipairs(rope.marker) do fixTable(m, baseDir) end
        else
            fixTable(rope.marker, baseDir)
        end
    end
end

-- Hook a convenient post-load to sweep rope objects (only if Winch exposes a place to do so)
if Winch ~= nil and Winch.loadFromXML ~= nil and not Winch.__RopeNormalizePatched then
    local _W_loadFromXML = Winch.loadFromXML
    Winch.loadFromXML = function(self, xmlFile, key, ...)
        local ok = _W_loadFromXML(self, xmlFile, key, ...)
        -- After ropes are loaded, try to locate and normalize them
        -- Common patterns: self.winchRopes (array), or self.winch.rope / self.rope
        local candidates = {}
        if type(self.winchRopes) == "table" then
            for _, r in ipairs(self.winchRopes) do table.insert(candidates, r) end
        end
        if type(self.winch) == "table" and type(self.winch.rope) == "table" then
            table.insert(candidates, self.winch.rope)
        end
        if type(self.rope) == "table" then
            table.insert(candidates, self.rope)
        end
        for _, r in ipairs(candidates) do
            normalizeRopeObjectPaths(self, r, xmlFile)
        end
        return ok
    end
    Winch.__RopeNormalizePatched = true
    _whpf_log("Patched Winch:loadFromXML to normalize paths on mainRope/setupRope/marker/treeHook after load")
end
