--[[
	en: 'Lock steering axle' Specialization for Farming Simulator 17
	en: This script allows you to lock an unlock the steering axle
	
	de: 'Lenkachse sperren' Spezialisierung für den Landwirtschafts Simulator 17
	de: Dieses Script erlaubt dir die Lenkachse zu sperren und zu entsperren.
	
	Author: 	Ifko[nator]
	Date:		21.04.2022
	Version:	3.4
	
	History:	v1.0 @07.05.2017 - initial implementation in FS 17
				------------------------------------------------------------------------------------------------
				v1.5 @10.05.2017 - add save function and support for the old wheel entry
				------------------------------------------------------------------------------------------------
									if the steering axle is unlocked, it will steer also in backwards direction
				------------------------------------------------------------------------------------------------
				v1.6 @13.05.2017 - fix for MP
				------------------------------------------------------------------------------------------------
				v1.7 @20.08.2017 - fix for joining players
				------------------------------------------------------------------------------------------------
				v1.8 @13.09.2017 - again fix for MP
				------------------------------------------------------------------------------------------------
				v2.0 @04.01.2019 - initial implementation in FS 19
				------------------------------------------------------------------------------------------------
				V2.1 @13.01.2019 - bug fix for BagLifter-Mod
				------------------------------------------------------------------------------------------------
				V2.2 @08.03.2019 - add support for trailers with an steering axle target node
				------------------------------------------------------------------------------------------------
				v2.3 @04.08.2021 - add automatic lock for the steering axles, if AutoDrive is driving backwards
				------------------------------------------------------------------------------------------------
				v3.0 @02.01.2022 - initial implementation in FS 22
				------------------------------------------------------------------------------------------------
				v3.1 @04.01.2022 - fix issues with the nardi cutter trailers
				------------------------------------------------------------------------------------------------
				v3.3 @16.03.2022 - fix for MP
				------------------------------------------------------------------------------------------------
				v3.4 @21.04.2022 - fix for patch 1.4 and higher
]]

LockSteeringAxles = {};
LockSteeringAxles.currentModName = "";
LockSteeringAxles.currentModDirectory = "";
LockSteeringAxles.AUTODRIVE_MOD_NAME = "";

for _, mod in pairs(g_modManager.mods) do
	if _G[tostring(mod.modName)].AutoDrive ~= nil then
		if g_modIsLoaded[tostring(mod.modName)] then	
			LockSteeringAxles.AUTODRIVE_MOD_NAME = mod.modName;

			break;
		end;
	end;
end;

for _, mod in pairs(g_modManager.mods) do
	if mod.title == "Lock steering axle" or mod.title == "Lenkachse sperren" or mod.title == "Blocage du différentiel sur pont avant" then
		if g_modIsLoaded[tostring(mod.modName)] then	
			LockSteeringAxles.currentModName = mod.modName;
			LockSteeringAxles.currentModDirectory = mod.modDir;

			break;
		end;
	end;
end;

function LockSteeringAxles.initSpecialization()
	local schemaSavegame = Vehicle.xmlSchemaSavegame;

	schemaSavegame:register(XMLValueType.BOOL, "vehicles.vehicle(?).lockSteeringAxles#lockSteeringAxle", "Steering Axle is locked.", false);
end;

function LockSteeringAxles.prerequisitesPresent(specializations)
	return SpecializationUtil.hasSpecialization(Attachable, specializations);
end;

function LockSteeringAxles.registerEventListeners(vehicleType)
	local functionNames = {
		"onLoad",
		"onUpdate",
		"saveToXMLFile",
		"onWriteStream",
		"onReadStream",
		"onRegisterActionEvents"
	};
	
	for _, functionName in ipairs(functionNames) do
		SpecializationUtil.registerEventListener(vehicleType, functionName, LockSteeringAxles);
	end;
end;

function LockSteeringAxles.registerFunctions(vehicleType)
	local functionNames = {
		"setSteeringAxleActive"
	};
	
	for _, functionName in ipairs(functionNames) do
		SpecializationUtil.registerFunction(vehicleType, functionName, LockSteeringAxles[functionName]);
	end;
end;

function LockSteeringAxles:onLoad(savegame)
	local specLockSteeringAxles = self.spec_lockSteeringAxles;
	local specWheels = self.spec_wheels;
	local specAttachable = self.spec_attachable;

	specLockSteeringAxles.foundSteeringAxle = false;
	specLockSteeringAxles.lockSteeringAxle = false;

	specLockSteeringAxles.l10nTexts = {};

	local l10nTexts = {
		"LOCK_STEERING_AXLE",
		"UNLOCK_STEERING_AXLE"
	};
	
	for _, l10nText in pairs(l10nTexts) do
		specLockSteeringAxles.l10nTexts[l10nText] = g_i18n:getText(l10nText, LockSteeringAxles.currentModName);
	end;

	if specWheels == nil or #specWheels.wheels == 0 or specAttachable.steeringAxleTargetNode ~= nil or specAttachable.steeringAxleReferenceComponentNode ~= nil then
		return;
	end;
	
	if self.xmlFile:hasProperty("vehicle.attachable.steeringAxleAngleScale") then
		specLockSteeringAxles.foundSteeringAxle = true;
			
		specAttachable.steeringAxleUpdateBackwards = true;
	end;

	if not specLockSteeringAxles.foundSteeringAxle then
		local wheelsKey = "vehicle.wheels.wheelConfigurations.wheelConfiguration(0).wheels";
		
		local wheelNumber = 0;
		
    	while true do
    	    local wheelKey = wheelsKey .. string.format(".wheel(%d)", wheelNumber);	
			
			if not self.xmlFile:hasProperty(wheelKey) then
				break;
			end;
			
			local steeringAxleRotMax = Utils.getNoNil(getXMLFloat(self.xmlFile.handle, wheelKey .. ".steeringAxle#rotMax"), 0);
		
			if steeringAxleRotMax ~= 0 then
				specLockSteeringAxles.foundSteeringAxle = true;
				
				specAttachable.steeringAxleUpdateBackwards = true;
				
				break;
			end;
			
			wheelNumber = wheelNumber + 1;
		end;
	end;

	if specLockSteeringAxles.foundSteeringAxle then
		if savegame ~= nil then
			specLockSteeringAxles.lockSteeringAxle = savegame.xmlFile:getValue(savegame.key .. ".lockSteeringAxles#lockSteeringAxle", specLockSteeringAxles.lockSteeringAxle);
		else
			specLockSteeringAxles.lockSteeringAxle = false;
		end;
		
		self:setSteeringAxleActive(specLockSteeringAxles.lockSteeringAxle, false);
	end;
end;

function LockSteeringAxles:onWriteStream(streamId, connection)
	if not connection:getIsServer() then
		local specLockSteeringAxles = self.spec_lockSteeringAxles;
		
		if specLockSteeringAxles.foundSteeringAxle and specLockSteeringAxles.lockSteeringAxle ~= nil then
			streamWriteBool(streamId, specLockSteeringAxles.lockSteeringAxle);
		end;
	end;
end;

function LockSteeringAxles:onReadStream(streamId, connection)
	if connection:getIsServer() then
		local specLockSteeringAxles = self.spec_lockSteeringAxles;
		
		if specLockSteeringAxles.foundSteeringAxle and specLockSteeringAxles.lockSteeringAxle ~= nil then	
			specLockSteeringAxles.lockSteeringAxle = streamReadBool(streamId);
			
			self:setSteeringAxleActive(specLockSteeringAxles.lockSteeringAxle, true);
		end;
	end;
end;

function LockSteeringAxles:onRegisterActionEvents(isActiveForInput)
	local specLockSteeringAxles = self.spec_lockSteeringAxles;
	
	if self.isClient and specLockSteeringAxles.foundSteeringAxle then        
		self:clearActionEventsTable(specLockSteeringAxles.actionEvents);
        
		if self:getIsActiveForInput(true) then
            local actionEventId;
            
			_, actionEventId = self:addActionEvent(specLockSteeringAxles.actionEvents, InputAction.TOGGLE_LOCK_STEERING_AXLE_BUTTON, self, LockSteeringAxles.toggleSteeringAxleActive, false, true, false, true, nil);
			
			g_inputBinding:setActionEventTextPriority(actionEventId, GS_PRIO_NORMAL);
			g_inputBinding:setActionEventTextVisibility(actionEventId, true);
			g_inputBinding:setActionEventActive(actionEventId, false);
		end;
	end;
end;

function LockSteeringAxles.toggleSteeringAxleActive(self, actionName, inputValue, callbackState, isAnalog)
	local specLockSteeringAxles = self.spec_lockSteeringAxles;
	
	self:setSteeringAxleActive(not specLockSteeringAxles.lockSteeringAxle, false);
end;

function LockSteeringAxles:onUpdate(dt, isActiveForInput, isSelected)
	local specLockSteeringAxles = self.spec_lockSteeringAxles;
	local specAttachable = self.spec_attachable;

	if specLockSteeringAxles.foundSteeringAxle then
		if LockSteeringAxles.AUTODRIVE_MOD_NAME ~= "" then
			local rootVehicle = self:getRootVehicle();
			
			if rootVehicle ~= nil then
				if rootVehicle.ad ~= nil then
					if rootVehicle.ad.isActive then
						specAttachable.steeringAxleUpdateBackwards = false;
					else
						specAttachable.steeringAxleUpdateBackwards = true;
					end;
				end;
			end;
		end;

		if self.isClient then	
			specAttachable.updateSteeringAxleAngle = not specLockSteeringAxles.lockSteeringAxle;
			
			local lockSteeringAxleButton = specLockSteeringAxles.actionEvents[InputAction.TOGGLE_LOCK_STEERING_AXLE_BUTTON];
			
			if lockSteeringAxleButton ~= nil and self:getIsActive() then
				local currentText = specLockSteeringAxles.l10nTexts.LOCK_STEERING_AXLE;

				g_inputBinding:setActionEventActive(lockSteeringAxleButton.actionEventId, true);
				
				if specLockSteeringAxles.lockSteeringAxle then
					currentText = specLockSteeringAxles.l10nTexts.UNLOCK_STEERING_AXLE;
				end;
				
				g_inputBinding:setActionEventText(lockSteeringAxleButton.actionEventId, currentText);
			end;
		end;

		if specLockSteeringAxles.lockSteeringAxle then
			if specAttachable.steeringAxleAngle ~= 0 then	
				if specAttachable.steeringAxleAngle > 0 then
					specAttachable.steeringAxleAngle = math.max(specAttachable.steeringAxleAngle - specAttachable.steeringAxleAngleSpeed / 2 * dt, 0);
				else
					specAttachable.steeringAxleAngle = math.min(specAttachable.steeringAxleAngle + specAttachable.steeringAxleAngleSpeed / 2 * dt, 0);
				end;
			end;
		end;
	end;

	if specAttachable.updateSteeringAxleAngle then
		local steeringAngle = 0;
		local baseVehicle = self:getSteeringAxleBaseVehicle();

		if (baseVehicle ~= nil or specAttachable.steeringAxleReferenceComponentNode ~= nil) and (self.movingDirection >= 0 or specAttachable.steeringAxleUpdateBackwards) then
			yRot = Utils.getYRotationBetweenNodes(self.steeringAxleNode, specAttachable.steeringAxleReferenceComponentNode or baseVehicle.steeringAxleNode);

			local scale = 1;

			if specAttachable.steeringAxleAngleScaleSpeedDependent then
				local startSpeed = specAttachable.steeringAxleAngleScaleStart;
				local endSpeed = specAttachable.steeringAxleAngleScaleEnd;

				scale = math.clamp(1 + (self:getLastSpeed() - startSpeed) * 1 / (startSpeed - endSpeed), 0, 1);
			end;

			steeringAngle = yRot * scale;
		elseif self:getLastSpeed() > 0.2 then
			steeringAngle = 0;
		end;

		if not self:getIsSteeringAxleAllowed() then
			steeringAngle = 0;
		end;

		if specAttachable.steeringAxleDistanceDelay > 0 then
			specAttachable.steeringAxleTargetAngleHistoryMoved = specAttachable.steeringAxleTargetAngleHistoryMoved + self.lastMovedDistance;

			if specAttachable.steeringAxleTargetAngleHistoryMoved > 0.1 then
				specAttachable.steeringAxleTargetAngleHistory[specAttachable.steeringAxleTargetAngleHistoryIndex] = steeringAngle;
				specAttachable.steeringAxleTargetAngleHistoryIndex = specAttachable.steeringAxleTargetAngleHistoryIndex + 1;

				if specAttachable.steeringAxleTargetAngleHistoryIndex > #specAttachable.steeringAxleTargetAngleHistory then
					specAttachable.steeringAxleTargetAngleHistoryIndex = 1;
				end;
			end;

			local lastIndex = specAttachable.steeringAxleTargetAngleHistoryIndex + 1;

			if lastIndex > #specAttachable.steeringAxleTargetAngleHistory then
				lastIndex = 1;
			end;

			specAttachable.steeringAxleTargetAngle = specAttachable.steeringAxleTargetAngleHistory[lastIndex];
		else
			specAttachable.steeringAxleTargetAngle = steeringAngle;
		end;

		local dir = math.sign(specAttachable.steeringAxleTargetAngle - specAttachable.steeringAxleAngle);
		local speed = specAttachable.steeringAxleAngleSpeed;

		if not self.finishedFirstUpdate then
			speed = 9999;
		end;

		if dir == 1 then
			specAttachable.steeringAxleAngle = math.min(specAttachable.steeringAxleAngle + dir * dt * speed, specAttachable.steeringAxleTargetAngle);
		else
			specAttachable.steeringAxleAngle = math.max(specAttachable.steeringAxleAngle + dir * dt * speed, specAttachable.steeringAxleTargetAngle);
		end;

		if specAttachable.steeringAxleTargetNode ~= nil and (self:getLastSpeed() > 0.25 or not self.finishedFirstUpdate) then
			local angle = nil;

			if specAttachable.steeringAxleTargetNodeRefAngle ~= nil then
				local alpha = math.clamp(specAttachable.steeringAxleAngle / specAttachable.steeringAxleTargetNodeRefAngle, -1, 1);

				if alpha >= 0 then
					angle = specAttachable.steeringAxleAngleMaxRot * alpha;
				else
					angle = specAttachable.steeringAxleAngleMinRot * -alpha;
				end;
			else
				angle = math.clamp(specAttachable.steeringAxleAngle, specAttachable.steeringAxleAngleMinRot, specAttachable.steeringAxleAngleMaxRot);
			end;

			setRotation(specAttachable.steeringAxleTargetNode, 0, angle * specAttachable.steeringAxleDirection, 0);

			self:setMovingToolDirty(specAttachable.steeringAxleTargetNode);
		end;
	end;
end;

function LockSteeringAxles:saveToXMLFile(xmlFile, key, usedModNames)
	local specLockSteeringAxles = self.spec_lockSteeringAxles;
	
	if specLockSteeringAxles.foundSteeringAxle and specLockSteeringAxles.lockSteeringAxle then
		xmlFile:setValue(key .. "#lockSteeringAxle", specLockSteeringAxles.lockSteeringAxle);
	end;
end;

function LockSteeringAxles:setSteeringAxleActive(lockSteeringAxle, noEventSend)
	local specLockSteeringAxles = self.spec_lockSteeringAxles;
	
	if lockSteeringAxle ~= specLockSteeringAxles.lockSteeringAxle then
		specLockSteeringAxles.lockSteeringAxle = lockSteeringAxle;
		
		if noEventSend == nil or noEventSend == false then
			if g_server ~= nil then
				g_server:broadcastEvent(LockSteeringAxlesEvent.new(self, lockSteeringAxle), nil, nil, self);
			else
				g_client:getServerConnection():sendEvent(LockSteeringAxlesEvent.new(self, lockSteeringAxle));
			end;
		end;
	end;
end;

--## MP Stuff

--## turn steering axle on/off

LockSteeringAxlesEvent = {};
LockSteeringAxlesEvent_mt = Class(LockSteeringAxlesEvent, Event);

InitEventClass(LockSteeringAxlesEvent, "LockSteeringAxlesEvent");

function LockSteeringAxlesEvent.emptyNew()
	local self = Event.new(LockSteeringAxlesEvent_mt);
    
	return self;
end;

function LockSteeringAxlesEvent.new(trailer, lockSteeringAxle)
	local self = LockSteeringAxlesEvent.emptyNew();
	
	self.trailer = trailer;
	self.steerAxleIsLocked = lockSteeringAxle;
	
	return self;
end;

function LockSteeringAxlesEvent:readStream(streamId, connection)
	self.trailer = NetworkUtil.readNodeObject(streamId);
	self.steerAxleIsLocked = streamReadBool(streamId);
	
    self:run(connection);
end;

function LockSteeringAxlesEvent:writeStream(streamId, connection)
	NetworkUtil.writeNodeObject(streamId, self.trailer);
	
	streamWriteBool(streamId, self.steerAxleIsLocked);
end;

function LockSteeringAxlesEvent:run(connection)
	if not connection:getIsServer() then
		g_server:broadcastEvent(LockSteeringAxlesEvent.new(self.trailer, self.steerAxleIsLocked), nil, connection, self.trailer);
	end;
	
    if self.trailer ~= nil then
        self.trailer:setSteeringAxleActive(self.steerAxleIsLocked, true);
	end;
end;